// SJJ Embedded Micro Solutions, LLC.
// Copyright  2009 SJJ Embedded Micro Solutions, LLC. All Rights Reserved
//  
// THIS SAMPLE CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, 
// INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.

/*****************************************************************************
*
* FILE NAME:    MM_AandF.c
*
* DESCRIPTION:    This is the main program module.
*
\*****************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include <rt.h>


#include "MM_AandF.h"

// global variables
  RTHANDLE hRootProcess;
  DWORD    dwKtickInUsecs;
// client thread declarations

  void     ClientThread1(void);
  RTHANDLE hClientThread1;

// module variables
struct _INIT {
  enum States
       { BEFORE_INIT,
         INIT_BUSY,
         INIT_DONE,
         CLEANUP_BUSY } state;
  RTHANDLE hMain;
  BOOL     bCataloged;
	// shared memory declarations

  PVOID    pSharedmem1;
  RTHANDLE hSharedmem1;

	// client thread declarations

  RTHANDLE hClientThread1;

}  strInit;

static void     Cleanup(void);

/*****************************************************************************
*
* FUNCTION:     main
*
* DESCRIPTION:
*  This is the main program module.
*  It creates shared memory objects, and all threads.
*  The main thread then waits for a deletion message,
*  and then all objects are deleted and the process terminates itself.
*
\*****************************************************************************/

void        main(void)
{
  SYSINFO   sysinfo;
  EVENTINFO eiEventInfo;
#ifdef _DEBUG
  printf("MM_AandF started\n");
#endif

  // obtain handle of root process (cannot fail)
  hRootProcess = GetRtThreadHandles(ROOT_PROCESS);

  // initialize the structure for cleaning up
  memset(&strInit, 0, sizeof(strInit));
  strInit.state = BEFORE_INIT;
  
  // get low level tick length in usecs
  if (!CopyRtSystemInfo(&sysinfo))
    Fail("Cannot copy system info");
  dwKtickInUsecs = 10000 / sysinfo.KernelTickRatio;
  if (dwKtickInUsecs == 0)
    Fail("Invalid low level tick length");

  // adjust process max priority (ignore error)
  SetRtProcessMaxPriority(NULL_RTHANDLE, 150);

  // obtain main thread's handle
  strInit.hMain = GetRtThreadHandles(THIS_THREAD);
  strInit.state = INIT_BUSY;

  // catalog the token of this process in the root process
  if (!Catalog(hRootProcess,
      GetRtThreadHandles(THIS_PROCESS),
      "MM_AandF"))
    Fail("Cannot catalog process name");
  strInit.bCataloged = TRUE;

//--
#if 0
  // allocate shared memory

  strInit.pSharedmem1 = AllocateRtMemory(4096);
  if (strInit.pSharedmem1 == NULL)
    Fail("Cannot allocate shared memory Sharedmem1");
  strInit.hSharedmem1 = CreateRtMemoryHandle(
            strInit.pSharedmem1, 4096);
  if (strInit.hSharedmem1 == BAD_RTHANDLE)
  {
    strInit.hSharedmem1 = NULL_RTHANDLE;
    Fail("Cannot create a handle for shared memory Sharedmem1");
  }
  if (!Catalog(NULL_RTHANDLE, strInit.hSharedmem1, "Sharedmem1"))
    Fail("Cannot catalog shared memory Sharedmem1");
#endif
//--

  // create client threads

  strInit.hClientThread1
          = CreateRtThread(180,
            ClientThread1,
            4096,
            0);
  if (strInit.hClientThread1 == BAD_RTHANDLE)
  {
    strInit.hClientThread1 = NULL_RTHANDLE;
    Fail("Cannot create client thread ClientThread1");
  }

  // indicate that initialization has finished
  strInit.state = INIT_DONE;
#ifdef _DEBUG
  printf("MM_AandF finished initialization\n");
#endif

  // wait for notification
  while (RtNotifyEvent(RT_SYSTEM_NOTIFICATIONS | RT_EXIT_NOTIFICATIONS,
      WAIT_FOREVER,
      &eiEventInfo))
  {
    switch(eiEventInfo.dwNotifyType)
    {
    case TERMINATE:
      // TODO: this process should terminate
      // cleanup the environment
      Cleanup();  // does not return

    case RT_CLIENT_DOWN:
      // TODO: on a NT host react to a RT client that has gone down
      break;

    case RT_CLIENT_UP:
      // TODO: on a NT host react to a RT client that has come back
      break;

    case NT_HOST_DOWN:
      // TODO: on a RT client react to a NT host that has gone down
      break;

    case NT_HOST_UP:
      // TODO: on a RT client react to a NT host that has come back
      break;

    case NT_BLUESCREEN:
      // TODO: react to a NT blue screen
      break;
    }
  }
  Fail("Notify failed");
}


/*****************************************************************************
*
* FUNCTION:     Fail
*
* PARAMETERS:   same parameters as expected by printf
*
* RETURNS:      does not return
*
* DESCRIPTION:
*  If in debug mode, prints the message.
*  Then the current process is killed ungraciously.
\*****************************************************************************/

void        Fail(
  LPSTR       lpszMessage, ...)
{
  EXCEPTION   eh;
  RTHANDLE    hDelMbx;
  DWORD     dwTerminate;

#ifdef _DEBUG
  va_list     ap;

  va_start(ap, lpszMessage);
  vprintf(lpszMessage, ap);
  va_end(ap);
  printf("\nError nr=%x\n", GetLastRtError());
#endif

  // make sure that exceptions are returned for inline handling
  GetRtExceptionHandlerInfo(THREAD_HANDLER, &eh);
  eh.ExceptionMode = 0;
  SetRtExceptionHandler(&eh);

  // if we had not started initializing yet, just get out
  if (strInit.state == BEFORE_INIT)
    exit(0);

  if (strInit.hMain == GetRtThreadHandles(THIS_THREAD))
  {
    // this is the main thread:
    // if we are busy initializing, then do Cleanup
    if (strInit.state == INIT_BUSY)
      Cleanup();  // does not return

    // this is the main thread, but we are not initializing:
    // just return
    return;
  }

  // this is not the main thread:
  // ask main thread to do cleanup
  // (allow some time to setup the deletion mailbox, ignore errors)
  hDelMbx = LookupRtHandle(NULL_RTHANDLE, "R?EXIT_MBOX", 5000);
  dwTerminate = TERMINATE;
  SendRtData(hDelMbx, &dwTerminate, 4);
  SuspendRtThread(NULL_RTHANDLE);

  // if we get here, suspend has failed
  while (1)
    RtSleep(655349);  // sleep for the maximum time
}


/*****************************************************************************
*
* FUNCTION:   Cleanup (local function)
*
* DESCRIPTION:
*  Remove all objects and other process side effects, 
*  as far as they have been created.
*
\*****************************************************************************/

void      Cleanup(void)
{
  // indicate that we are cleaning up
  strInit.state = CLEANUP_BUSY;

  // kill client threads

  if (strInit.hClientThread1 != NULL_RTHANDLE)
    if (!DeleteRtThread(strInit.hClientThread1))
      Fail("Cannot delete client thread ClientThread1");

  // free shared memory

  if (strInit.hSharedmem1 != NULL_RTHANDLE)
    if (!DeleteRtMemoryHandle(strInit.hSharedmem1))
      Fail("Cannot free shared memory handle Sharedmem1");
  
  if (strInit.pSharedmem1 != NULL)
    if (!FreeRtMemory(strInit.pSharedmem1))
      Fail("Cannot free shared memory Sharedmem1");

  // remove our name from the root process
  if (strInit.bCataloged)
    if (!UncatalogRtHandle(hRootProcess, "MM_AandF"))
      Fail("Cannot remove my own name");

#ifdef _DEBUG
  printf("MM_AandF finished cleaning up\n");
#endif

  // lie down
  exit(0);
}
